import * as state from './state.js';
import * as ui from './ui.js';
import * as ipc from './ipc.js';
import { playSound, navigateSound, confirmSound, cancelSound, anyModalOpen, applyTranslationsToDOM, t, backgroundMusic, attractMusicPlayer, fadeOutAndStop, shuffleArray } from './utils.js';

function startAttractMode() {
    if (state.appSettings.attractModeEnabled !== 'true' || anyModalOpen() || state.isProcessActive) {
        return;
    }

    ui.attractModeOverlay.classList.add('visible');
    fadeOutAndStop(backgroundMusic);

    const customBgm = state.availableBgmTracks.filter(track =>
        track.path !== null && track.name !== 'Music Off' && track.name !== 'Default'
    );

    if (customBgm.length > 0) {
        const playlist = shuffleArray(customBgm);
        state.setAttractPlaylist(playlist);
        state.setAttractPlaylistIndex(0);

        const playNextAttractSong = () => {
            if (state.attractPlaylist.length === 0) return;
            
            const track = state.attractPlaylist[state.attractPlaylistIndex];
            attractMusicPlayer.src = `file:///${track.path.replace(/\\/g, '/')}`;
            attractMusicPlayer.volume = state.appSettings.bgmVolumeLevel / 10 * 0.5;
            attractMusicPlayer.play().catch(e => console.error("Attract BGM failed to play:", e));

            state.setAttractPlaylistIndex((state.attractPlaylistIndex + 1) % state.attractPlaylist.length);
        };

        attractMusicPlayer.onended = playNextAttractSong;
        playNextAttractSong();
    }

    const runSonicSequence = () => {
        ui.attractSonic.className = 'sonic-run-right';
        setTimeout(() => {
            ui.attractSonic.className = 'sonic-run-left';
        }, 3000); // 1s run + 2s pause
        setTimeout(() => {
            ui.attractSonic.className = '';
        }, 4000); // 3s delay + 1s run
    };

    setTimeout(() => {
        runSonicSequence();
        const interval = setInterval(runSonicSequence, 20000);
        state.setAttractSonicInterval(interval);
    }, 10000);
}

export function resetInactivityTimer() {
    ui.attractModeOverlay.classList.remove('visible');
    clearTimeout(state.inactivityTimer);
    clearInterval(state.attractSonicInterval);
    state.setAttractSonicInterval(null);
    ui.attractSonic.className = '';

    if (!attractMusicPlayer.paused) {
        fadeOutAndStop(attractMusicPlayer);
        if (state.appSettings.bgmPath && state.appSettings.bgmPath !== "off") {
            backgroundMusic.volume = state.appSettings.bgmVolumeLevel / 10 * 0.5;
            backgroundMusic.currentTime = 0;
            backgroundMusic.play().catch(e => {});
        }
    }

    if (backgroundMusic.paused && state.appSettings.bgmPath && state.appSettings.bgmPath !== "off" && !state.isProcessActive && attractMusicPlayer.paused) {
        backgroundMusic.play().catch(e => {});
    }

    if (state.appSettings.attractModeEnabled === 'true' && !anyModalOpen() && !state.isProcessActive) {
        const timer = setTimeout(startAttractMode, state.ATTRACT_MODE_TIMEOUT);
        state.setInactivityTimer(timer);
    }
}


const letterJumpOverlay = document.getElementById('letter-jump-overlay');
const letterJumpIndicator = document.getElementById('letter-jump-indicator');
let letterJumpTimeout = null;


function dispatchKeyEvent(keyName) {
    window.dispatchEvent(new KeyboardEvent('keydown', { key: keyName }));
}


function getCharacterGroup(char) {
    if (!char) return '#';
    return (char >= '0' && char <= '9') ? '#' : char.toUpperCase();
}

function jumpToLetter(direction) {
    const totalGames = state.games.length;
    if (totalGames <= 1) return;

    
    const showLetterIndicator = (targetIndex) => {
        if (targetIndex < 0 || targetIndex >= state.games.length) return;
        const targetLetter = getCharacterGroup(state.games[targetIndex].baseName.charAt(0));

        if (letterJumpTimeout) {
            clearTimeout(letterJumpTimeout);
        }
        letterJumpIndicator.textContent = targetLetter;
        letterJumpOverlay.classList.add('visible');

        letterJumpTimeout = setTimeout(() => {
            letterJumpOverlay.classList.remove('visible');
        }, 600); 
    };

    playSound(navigateSound);

    const currentGame = state.games[state.selectedIndex];
    const currentInitial = getCharacterGroup(currentGame.baseName.charAt(0));

    if (direction < 0) { 
        const firstIndexOfCurrentLetter = state.games.findIndex(g => getCharacterGroup(g.baseName.charAt(0)) === currentInitial);

        if (state.selectedIndex > firstIndexOfCurrentLetter) {
            showLetterIndicator(firstIndexOfCurrentLetter);
            ui.updateSelection(firstIndexOfCurrentLetter);
            return;
        }

        let targetIndex = -1;
        for (let i = state.selectedIndex - 1; i >= 0; i--) {
            const initial = getCharacterGroup(state.games[i].baseName.charAt(0));
            if (initial !== currentInitial) {
                const prevInitial = initial;
                targetIndex = state.games.findIndex(g => getCharacterGroup(g.baseName.charAt(0)) === prevInitial);
                break;
            }
        }
        if (targetIndex !== -1) {
            showLetterIndicator(targetIndex);
            ui.updateSelection(targetIndex);
        } else {
            const lastInitial = getCharacterGroup(state.games[totalGames-1].baseName.charAt(0));
            const firstIndexOfLastLetter = state.games.findIndex(g => getCharacterGroup(g.baseName.charAt(0)) === lastInitial);
            showLetterIndicator(firstIndexOfLastLetter);
            ui.updateSelection(firstIndexOfLastLetter);
        }

    } else { 
        let targetIndex = -1;
        for (let i = state.selectedIndex + 1; i < totalGames; i++) {
            if (getCharacterGroup(state.games[i].baseName.charAt(0)) !== currentInitial) {
                targetIndex = i;
                break;
            }
        }
        if (targetIndex !== -1) {
            showLetterIndicator(targetIndex);
            ui.updateSelection(targetIndex);
        } else {
            showLetterIndicator(0);
            ui.updateSelection(0);
        }
    }
}


function handleGridInput(key) {
    let currentIndex = state.selectedIndex;
    let newIndex = currentIndex;
    const totalGames = state.games.length;
    if (totalGames === 0) return;

    if (key === 'ArrowRight') {
        newIndex = (currentIndex + 1) % totalGames;
    } else if (key === 'ArrowLeft') {
        
        if (currentIndex % state.GRID_COLUMNS === 0) {
            playSound(navigateSound);
            state.setFocusedElement('actions');
            state.setActionIconIndex(0); 
            ui.updateActionIconSelection();
            return;
        }
        newIndex = (currentIndex - 1 + totalGames) % totalGames;
    } else if (key === 'ArrowDown') {
        newIndex = Math.min(currentIndex + state.GRID_COLUMNS, totalGames - 1);
    } else if (key === 'ArrowUp') {
        newIndex = Math.max(currentIndex - state.GRID_COLUMNS, 0);
    } else if (key === 'Backspace' || key === 'Escape') {
        ui.openSystemOptionsMenu();
        return;
    }
    
    if (newIndex !== currentIndex) {
        ui.updateSelection(newIndex);
    }
}

function handleActionIconInput(key) {
    if (key === 'ArrowUp' || key === 'ArrowDown') {
        playSound(navigateSound);
        const direction = key === 'ArrowUp' ? -1 : 1;
        const newIndex = (state.actionIconIndex + direction + 3) % 3; 
        state.setActionIconIndex(newIndex);
        ui.updateActionIconSelection();
    } else if (key === 'ArrowRight') {
        playSound(navigateSound);
        state.setFocusedElement('grid');
 
        ui.updateSelection(state.selectedIndex, false);
    } else if (key === 'Enter') {
        playSound(confirmSound);
        if (state.actionIconIndex === 0) {
            ui.openConsoleMenu();
        } else if (state.actionIconIndex === 1) { 
            ui.openLobbyMenu('actionIcon'); 
        } else if (state.actionIconIndex === 2) {
            ui.openSystemOptionsMenu();
        }
    } else if (key === 'Escape' || key === 'Backspace') {
        playSound(cancelSound);
        state.setFocusedElement('grid');
        ui.updateSelection(state.selectedIndex, false);
    }
}


function handleSimpleMenu(key, elements, index, setIndexFunc) {
    if (key === 'ArrowUp' || key === 'ArrowDown') {
        playSound(navigateSound);
        const direction = key === 'ArrowUp' ? -1 : 1;
        const newIndex = (index + direction + elements.length) % elements.length;
        setIndexFunc(newIndex);
        elements.forEach((el, i) => el.classList.toggle('active', i === newIndex));
    } else if (key === 'Enter') {
        playSound(confirmSound);
        elements[index].click();
    }
}

function handleOptionsItem(key, index) {
    if (key !== 'ArrowLeft' && key !== 'ArrowRight') return;
    playSound(navigateSound);
    const direction = key === 'ArrowRight' ? 1 : -1;
    
    if (state.isIngameOptionsOpen) {
        switch (index) {
            case 0:
                state.appSettings.savestate_auto_index = state.appSettings.savestate_auto_index === 'true' ? "false" : "true";
                break;
            case 1: 
                const aspects = Object.keys(state.aspectValues);
                let currentAspectIndex = aspects.indexOf(ui.aspectRatioDisplay.textContent);
                currentAspectIndex = (currentAspectIndex + direction + aspects.length) % aspects.length;
                state.appSettings.aspect_ratio_index = state.aspectValues[aspects[currentAspectIndex]];
                break;
            case 2: state.appSettings.video_fullscreen = state.appSettings.video_fullscreen === 'true' ? 'false' : 'true'; break;
            case 3: state.appSettings.input_overlay_enable = state.appSettings.input_overlay_enable === 'true' ? 'false' : 'true'; break;
            case 4:
                const strengths = Object.keys(state.strengthValues);
                let currentStrengthIndex = strengths.findIndex(s => t(`configmenu_strength_${s.toLowerCase()}`) === ui.scanlinesStrengthDisplay.textContent);
                currentStrengthIndex = (currentStrengthIndex + direction + strengths.length) % strengths.length;
                state.appSettings.input_overlay_opacity = state.strengthValues[strengths[currentStrengthIndex]];
                break;
            case 5: 
                const isRwEnabled = state.appSettings.rewind_enable === 'true';
                state.appSettings.rewind_enable = isRwEnabled ? 'false' : 'true';
                state.appSettings.input_hold_fast_forward_axis = isRwEnabled ? 'nul' : '+5';
                break;
        }
        ui.populateIngameOptionsMenu();
    }
    else if (state.isGeneralOptionsOpen) {
        switch(index) {
            case 0: state.appSettings.startFullscreen = !state.appSettings.startFullscreen; ipc.setFullscreen(state.appSettings.startFullscreen); break;
            case 1:
                const langIndex = state.availableLanguages.findIndex(l => l.code === state.appSettings.language);
                const nextLangIndex = (langIndex + direction + state.availableLanguages.length) % state.availableLanguages.length;
                const newLangCode = state.availableLanguages[nextLangIndex].code;
                state.appSettings.language = newLangCode;
                state.appSettings.user_language = state.RETROARCH_LANG_MAP[newLangCode] || '0';
                ipc.loadTranslations(newLangCode).then(translations => {
                    applyTranslationsToDOM(translations);
                    ui.populateGeneralMenu();
                });
                return; 
            case 2: 
                // Toggle Logo Region
                const regionIndex = state.availableRegions.indexOf(state.appSettings.logoRegion);
                const nextRegionIndex = (regionIndex + direction + state.availableRegions.length) % state.availableRegions.length;
                state.appSettings.logoRegion = state.availableRegions[nextRegionIndex];
                ui.updateMainLogo();
                ui.populateGeneralMenu();
                break;
            case 3:
                state.appSettings.attractModeEnabled = state.appSettings.attractModeEnabled === 'true' ? 'false' : 'true';
                resetInactivityTimer();
                break;
            case 4: 
                state.appSettings.bgmVolumeLevel = Math.max(0, Math.min(10, state.appSettings.bgmVolumeLevel + direction));
                backgroundMusic.volume = state.appSettings.bgmVolumeLevel / 10 * 0.5;
                break;
            case 5: 
                ui.changeBgm(direction);
                break;
        }
        ui.populateGeneralMenu();
    }
    else if (state.isNetworkFeaturesOpen) {
        if (index === 2) {
             state.appSettings.cheevos_enable = state.appSettings.cheevos_enable === 'true' ? "false" : "true";
             ui.populateNetworkMenu();
        } else if (index === 4) {
            const newIndex = (state.currentRelayIndex + direction + state.relayOptions.length) % state.relayOptions.length;
            state.setCurrentRelayIndex(newIndex);
            ui.updateRelayServerDisplay();
        }
    }
}

function handleSaveStatesMenuInput(key) {
    if (state.saveStatesNavRow === 0) { 
        if (key === 'ArrowLeft' || key === 'ArrowRight') {
            playSound(navigateSound);
            const direction = key === 'ArrowLeft' ? -1 : 1;
            let newIndex = state.saveStatesSlotSelectionIndex + direction;
            newIndex = Math.max(0, Math.min(newIndex, 3));
            state.setSaveStatesSlotSelectionIndex(newIndex);
            ui.updateSaveStatesSelection();
        } else if (key === 'Enter') {
            if (state.currentSaveStates[state.saveStatesSlotSelectionIndex]) {
                playSound(confirmSound);
                state.setSaveStatesNavRow(1);
                state.setSaveStatesActionSelectionIndex(0); 
                ui.updateSaveStatesSelection();
            }
        }
    } else { 
        if (key === 'ArrowLeft' || key === 'ArrowRight') {
            playSound(navigateSound);
            const newIndex = (state.saveStatesActionSelectionIndex + 1) % 2;
            state.setSaveStatesActionSelectionIndex(newIndex);
            ui.updateSaveStatesSelection();
        } else if (key === 'ArrowUp') {
            playSound(navigateSound);
            state.setSaveStatesNavRow(0);
            ui.updateSaveStatesSelection();
        } else if (key === 'Enter') {
            playSound(confirmSound);
            ui.saveStatesActions[state.saveStatesActionSelectionIndex].click();
        }
    }
}

let previousButtonStates = {};
function checkAnyButton(gamepads, buttonIndex, stateKey, action) {
    const isPressed = gamepads.some(gp => gp && gp.buttons[buttonIndex]?.pressed);
    if (isPressed && !previousButtonStates[stateKey]) {
        resetInactivityTimer();
        action();
    }
    previousButtonStates[stateKey] = isPressed;
}


function pollGamepads() {
    if (!document.hasFocus()) { requestAnimationFrame(pollGamepads); return; }
    
    const gamepad = Array.from(navigator.getGamepads()).find(gp => gp);
    if (!gamepad) { requestAnimationFrame(pollGamepads); return; }

 
    const wasInAttractModeGamepad = ui.attractModeOverlay.classList.contains('visible');
    const isAnyInputActive = gamepad.buttons.some(b => b.pressed) || gamepad.axes.some(a => Math.abs(a) > 0.5);

    if (wasInAttractModeGamepad && isAnyInputActive) {
        resetInactivityTimer();
        requestAnimationFrame(pollGamepads); 
        return;
    }

    const A_BUTTON = 0;
    const B_BUTTON = 1;
    const LB_BUTTON = 4;
    const RB_BUTTON = 5;
    const SELECT_BUTTON = 8;
    const START_BUTTON = 9;
    const DPAD_UP = 12;
    const DPAD_DOWN = 13;
    const DPAD_LEFT = 14;
    const DPAD_RIGHT = 15;
    
    const up = gamepad.buttons[DPAD_UP]?.pressed || gamepad.axes[1] < -0.5;
    const down = gamepad.buttons[DPAD_DOWN]?.pressed || gamepad.axes[1] > 0.5;
    const left = gamepad.buttons[DPAD_LEFT]?.pressed || gamepad.axes[0] < -0.5;
    const right = gamepad.buttons[DPAD_RIGHT]?.pressed || gamepad.axes[0] > 0.5;

    if (state.isProcessActive) {
        checkAnyButton([gamepad], B_BUTTON, 'B', () => ipc.cancelEmuProcess());
    } 
    else if (state.isNetplayUsernameOpen || state.isRaLoginOpen) {
        checkAnyButton([gamepad], B_BUTTON, 'B', () => {
            playSound(cancelSound);
            if (state.isNetplayUsernameOpen) ui.closeNetplayUsernameModal();
            else ui.closeRaLoginModal();
        });
    }
    else if (anyModalOpen()) {
       
        processDirection(up, 'Up', 'ArrowUp');
        processDirection(down, 'Down', 'ArrowDown');
        processDirection(left, 'Left', 'ArrowLeft');
        processDirection(right, 'Right', 'ArrowRight');
        checkAnyButton([gamepad], A_BUTTON, 'A', () => dispatchKeyEvent('Enter'));
        checkAnyButton([gamepad], B_BUTTON, 'B', () => dispatchKeyEvent('Escape'));
    }

    else {
        if (state.focusedElement === 'actions') {
            processDirection(up, 'Up', 'ArrowUp');
            processDirection(down, 'Down', 'ArrowDown');
            processDirection(right, 'Right', 'ArrowRight');
            checkAnyButton([gamepad], A_BUTTON, 'A', () => dispatchKeyEvent('Enter'));
            checkAnyButton([gamepad], B_BUTTON, 'B', () => dispatchKeyEvent('Escape'));
        } else {
            processDirection(up, 'Up', 'ArrowUp');
            processDirection(down, 'Down', 'ArrowDown');
            processDirection(left, 'Left', 'ArrowLeft');
            processDirection(right, 'Right', 'ArrowRight');
            checkAnyButton([gamepad], A_BUTTON, 'A', () => dispatchKeyEvent('Enter'));
            checkAnyButton([gamepad], B_BUTTON, 'B', () => dispatchKeyEvent('Escape'));
            checkAnyButton([gamepad], START_BUTTON, 'Start', () => dispatchKeyEvent('Enter'));
            checkAnyButton([gamepad], SELECT_BUTTON, 'Select', () => dispatchKeyEvent('Escape'));
            checkAnyButton([gamepad], LB_BUTTON, 'LB', () => jumpToLetter(-1));
            checkAnyButton([gamepad], RB_BUTTON, 'RB', () => jumpToLetter(1));
        }
    }
    
    requestAnimationFrame(pollGamepads);
}

const buttonPressedTime = {};
const lastInputTime = {};

function processDirection(isActive, key, eventKey) {
    const now = Date.now();
    if (isActive) {
        if (!previousButtonStates[key]) {
            resetInactivityTimer();
            dispatchKeyEvent(eventKey);
            previousButtonStates[key] = true;
            buttonPressedTime[key] = now;
            lastInputTime[key] = now;
        } else if (now - buttonPressedTime[key] > state.INITIAL_REPEAT_DELAY) {
            if (now - lastInputTime[key] > state.REPEAT_DELAY) {
                dispatchKeyEvent(eventKey);
                lastInputTime[key] = now;
            }
        }
    } else {
        previousButtonStates[key] = false;
    }
}


export function initializeInput() {
    window.addEventListener('keydown', (e) => {
        const wasInAttractModeKeyboard = ui.attractModeOverlay.classList.contains('visible');
        resetInactivityTimer();

   
        if (wasInAttractModeKeyboard) {
            return;
        }

        if (state.isProcessActive) {
            if (e.key === 'Escape' || e.key === 'Backspace') {
                ipc.cancelEmuProcess();
            }
            return; 
        }

        if (state.isNetplayUsernameOpen || state.isRaLoginOpen) {
            if (e.key === 'Escape') {
                playSound(cancelSound);
                if (state.isNetplayUsernameOpen) ui.closeNetplayUsernameModal();
                else ui.closeRaLoginModal();
                return;
            } else if (e.key === 'Backspace') {
                if (document.activeElement.tagName !== 'INPUT') {
                    playSound(cancelSound);
                    if (state.isNetplayUsernameOpen) ui.closeNetplayUsernameModal();
                    else ui.closeRaLoginModal();
                }
            }
        }
        else if (state.isLobbyOpen) {
            if (e.key === 'Escape' || e.key === 'Backspace') { ui.closeLobbyMenu(); return; }
            if (state.lobbyGames.length > 0) {
                 if (e.key === 'ArrowUp' || e.key === 'ArrowDown') {
                    playSound(navigateSound);
                    const direction = e.key === 'ArrowUp' ? -1 : 1;
                    const newIndex = (state.lobbySelectionIndex + direction + state.lobbyGames.length) % state.lobbyGames.length;
                    state.setLobbySelectionIndex(newIndex);
                    ui.updateLobbySelection();
                } else if (e.key === 'Enter') {
                    playSound(confirmSound);
                    const selectedGame = state.lobbyGames[state.lobbySelectionIndex];
                    // FIX: Ensure join path always points to Sega Genesis folder, regardless of current view
                    const currentGamesDir = state.APP_PATHS.gamesDir.replace(/\\/g, '/');
                    const gamesRoot = currentGamesDir.substring(0, currentGamesDir.lastIndexOf('/'));
                    const genesisGamesDir = `${gamesRoot}/Sega Genesis`;
                    
                    const gamePath = `${genesisGamesDir}/${selectedGame.game_name}.zip`;
                    
                    ipc.joinNetplayGame({
                        ...selectedGame,
                        gamePath
                    });
                    ui.showProgressOverlay(t('progress_joining_game', { gameName: selectedGame.game_name }));
                }
            }
        }
        else if (state.isStartMenuOpen) {
            if (e.key === 'Escape' || e.key === 'Backspace') {
                ui.closeStartGameMenu();
                return;
            }
            if (e.key === 'ArrowUp' || e.key === 'ArrowDown') {
                playSound(navigateSound);
                const direction = e.key === 'ArrowUp' ? -1 : 1;
                let newIndex = state.startMenuSelectionIndex;
                const options = Array.from(ui.startMenuOptions);
                let attempts = 0;
                do {
                    newIndex = (newIndex + direction + options.length) % options.length;
                    attempts++;
                } while (options[newIndex].style.display === 'none' && attempts < options.length);

                if (newIndex !== state.startMenuSelectionIndex) {
                    state.setStartMenuSelectionIndex(newIndex);
                    options.forEach((el, i) => el.classList.toggle('active', i === newIndex));
                }
            } else if (e.key === 'Enter') {
                playSound(confirmSound);
                ui.startMenuOptions[state.startMenuSelectionIndex].click();
            }
        }
        else if (state.isConsoleMenuOpen) {
            if (e.key === 'Escape' || e.key === 'Backspace') { ui.closeConsoleMenu(); return; }
            handleSimpleMenu(e.key, ui.consoleMenuList, state.consoleMenuSelectionIndex, state.setConsoleMenuSelectionIndex);
        }
        else if (state.isSystemOptionsMenuOpen) {
            if (e.key === 'Escape' || e.key === 'Backspace') { ui.closeSystemOptionsMenu(); return; }
            handleSimpleMenu(e.key, ui.systemOptionsList, state.systemOptionsSelectionIndex, state.setSystemOptionsSelectionIndex);
        }
        else if (state.isIngameOptionsOpen) {
            if (e.key === 'Escape' || e.key === 'Backspace') { ui.closeIngameOptionsMenu(); return; }
            handleSimpleMenu(e.key, ui.ingameOptionsList, state.ingameOptionsSelectionIndex, state.setIngameOptionsSelectionIndex);
            handleOptionsItem(e.key, state.ingameOptionsSelectionIndex);
        }
        else if (state.isGeneralOptionsOpen) {
            if (e.key === 'Escape' || e.key === 'Backspace') { ui.closeGeneralOptionsMenu(); return; }
            handleSimpleMenu(e.key, ui.generalOptionsList, state.generalOptionsSelectionIndex, state.setGeneralOptionsSelectionIndex);
            handleOptionsItem(e.key, state.generalOptionsSelectionIndex);
        }
        else if (state.isNetworkFeaturesOpen) {
            if (e.key === 'Escape' || e.key === 'Backspace') { ui.closeNetworkFeaturesMenu(); return; }
            handleSimpleMenu(e.key, ui.networkFeaturesList, state.networkFeaturesSelectionIndex, state.setNetworkFeaturesSelectionIndex);
            handleOptionsItem(e.key, state.networkFeaturesSelectionIndex);
        }
         else if (state.isPowerMenuOpen) {
            if (e.key === 'Escape' || e.key === 'Backspace') { ui.closePowerMenu(); return; }
            handleSimpleMenu(e.key, ui.powerMenuList, state.powerMenuSelectionIndex, state.setPowerMenuSelectionIndex);
        }
         else if (state.isSaveStatesMenuOpen) {
            if (e.key === 'Escape' || e.key === 'Backspace') { ui.closeSaveStatesMenu(); return; }
            handleSaveStatesMenuInput(e.key);
        }
        
        else {
            if (state.focusedElement === 'actions') {
                handleActionIconInput(e.key);
            } else { 
                if (e.key === 'Enter') { 
                    ui.openStartGameMenu(); 
                    return; 
                }
                handleGridInput(e.key);
            }
        }
    });

 
    document.getElementById('start-game-button').addEventListener('click', ui.openStartGameMenu);
    document.getElementById('system-options-button').addEventListener('click', ui.openSystemOptionsMenu);

    document.getElementById('start-game-offline-button').addEventListener('click', () => { const gamePath = `${state.APP_PATHS.gamesDir}/${state.games[state.selectedIndex].zipName}`.replace(/\\/g, '/'); ipc.launchGame({ gamePath }); ui.showProgressOverlay(t('progress_initializing')); });
    document.getElementById('start-game-netplay-button').addEventListener('click', () => { const gamePath = `${state.APP_PATHS.gamesDir}/${state.games[state.selectedIndex].zipName}`.replace(/\\/g, '/'); ipc.netplayHost({ gamePath }); ui.showProgressOverlay(t('progress_starting_netplay')); });
    document.getElementById('open-savestates-menu-button').addEventListener('click', ui.openSaveStatesMenu);

    document.getElementById('open-console-menu-button').addEventListener('click', () => { ui.closeSystemOptionsMenu(false); ui.openConsoleMenu(); });
    document.getElementById('open-ingame-options-button').addEventListener('click', ui.openIngameOptionsMenu);
    document.getElementById('open-general-options-button').addEventListener('click', ui.openGeneralOptionsMenu);
    document.getElementById('open-network-features-button').addEventListener('click', ui.openNetworkFeaturesMenu);
    document.getElementById('open-power-menu-button').addEventListener('click', ui.openPowerMenu);

    document.getElementById('restore-defaults-button').addEventListener('click', () => { playSound(confirmSound); Object.assign(state.appSettings, state.IN_GAME_DEFAULTS); ui.populateIngameOptionsMenu(); });
    
    document.getElementById('open-lobby-button').addEventListener('click', () => ui.openLobbyMenu('networkMenu'));
    document.getElementById('ra-login-button').addEventListener('click', ui.openRaLoginModal);
    document.getElementById('netplay-username-button').addEventListener('click', ui.openNetplayUsernameModal);
    
    document.getElementById('download-collection-button').addEventListener('click', () => { 
        ui.closeNetworkFeaturesMenu(false); 
        ipc.downloadCollection(state.currentPlatform.id); 
        ui.showProgressOverlay(t('progress_preparing_download')); 
    });

    document.getElementById('exit-app-button').addEventListener('click', () => ipc.exitApp());
    document.getElementById('suspend-pc-button').addEventListener('click', () => { ui.closePowerMenu(false); ipc.suspendPc(); });
    document.getElementById('power-off-pc-button').addEventListener('click', () => { ui.closePowerMenu(false); ipc.powerOffPc(); });

    document.getElementById('username-save-button').addEventListener('click', () => { playSound(confirmSound); state.appSettings.netplayUsername = ui.usernameInput.value || "Player1"; ipc.saveAllSettings(state.appSettings); ui.closeNetplayUsernameModal(); });
    document.getElementById('username-cancel-button').addEventListener('click', () => { playSound(cancelSound); ui.closeNetplayUsernameModal(); });
    document.getElementById('ra-login-save-button').addEventListener('click', () => { playSound(confirmSound); state.appSettings.cheevos_username = ui.raUsernameInput.value; state.appSettings.cheevos_password = ui.raPasswordInput.value; ipc.saveAllSettings(state.appSettings); ui.closeRaLoginModal(); });
    document.getElementById('ra-login-cancel-button').addEventListener('click', () => { playSound(cancelSound); ui.closeRaLoginModal(); });
    
    // Save States Actions
    ui.saveStatesActions[0].addEventListener('click', () => { // Load
        const stateData = state.currentSaveStates[state.saveStatesSlotSelectionIndex];
        if (!stateData) return;
        
        ui.closeSaveStatesMenu(false);
        
        const gamePath = `${state.APP_PATHS.gamesDir}/${state.games[state.selectedIndex].zipName}`.replace(/\\/g, '/');
        ipc.launchGame({ gamePath, stateSlot: stateData.fileSlot });
        ui.showProgressOverlay(t('progress_loading_savestate'));
    });
    ui.saveStatesActions[1].addEventListener('click', async () => { // Delete
        const stateData = state.currentSaveStates[state.saveStatesSlotSelectionIndex];
        if (!stateData) return;
        
        const gameBaseName = state.games[state.selectedIndex].zipName.replace(/\.(zip|chd)$/i, '');
        await ipc.manageSaveState({ action: 'delete', gameBaseName, slot: stateData.fileSlot });
        await ui.populateSaveStatesMenu();
        ui.updateSaveStatesSelection();
        delete state.saveStateCache[gameBaseName];
        ui.updateSaveStateIcon(state.selectedIndex);
    });

    
    ui.actionIconGear.addEventListener('click', () => {
        state.setFocusedElement('actions');
        state.setActionIconIndex(2);
        ui.updateActionIconSelection();
        ui.openSystemOptionsMenu();
    });
    ui.actionIconWifi.addEventListener('click', () => {
        state.setFocusedElement('actions');
        state.setActionIconIndex(1);
        ui.updateActionIconSelection();
        ui.openLobbyMenu('actionIcon');
    });
    ui.actionIconConsole.addEventListener('click', () => {
        state.setFocusedElement('actions');
        state.setActionIconIndex(0);
        ui.updateActionIconSelection();
        ui.openConsoleMenu();
    });

    
    ui.consoleMenuList.forEach(btn => {
        btn.addEventListener('click', () => {
            playSound(confirmSound);
            const platformId = btn.getAttribute('data-platform-id');
            const platformKey = Object.keys(state.PLATFORMS).find(k => k === platformId);
            ui.switchPlatform(platformKey);
            ui.closeConsoleMenu(false);
        });
    });


    requestAnimationFrame(pollGamepads);
}